from __future__ import print_function
import torch
import torch.nn as nn
import torch.nn.functional as F # Import F for functional calls

class SupConLoss(nn.Module):
    """Supervised Contrastive Learning: https://arxiv.org/pdf/2004.11362.pdf."""
    def __init__(self, temperature=0.07, contrast_mode='all',
                 base_temperature=0.07):
        super(SupConLoss, self).__init__()
        self.temperature = temperature
        self.contrast_mode = contrast_mode
        self.base_temperature = base_temperature

    def forward(self, features, labels=None, mask=None):
        """
        Compute supervised contrastive loss.
        Args:
            features: hidden vector of shape [bsz, n_views, ...].
            labels: ground truth of shape [bsz].
            mask: contrastive mask of shape [bsz, bsz], if set, labels will be ignored.
        Returns:
            A loss scalar.
        """
        device = (torch.device('cuda')
                  if features.is_cuda
                  else torch.device('cpu'))

        if len(features.shape) < 3:
            raise ValueError('`features` needs to be [bsz, n_views, ...], '
                             'at least 3 dimensions are required')
        if len(features.shape) > 3:
            features = features.view(features.shape[0], features.shape[1], -1)

        batch_size = features.shape[0]
        if labels is not None and mask is not None:
            raise ValueError('Cannot define both `labels` and `mask`')
        elif labels is None and mask is None:
            mask = torch.eye(batch_size, dtype=torch.float32).to(device)
        elif labels is not None:
            labels = labels.contiguous().view(-1, 1)
            if labels.shape[0] != batch_size:
                raise ValueError('Num of labels does not match num of features')
            mask = torch.eq(labels, labels.T).float().to(device)
        else:
            mask = mask.float().to(device)

        contrast_count = features.shape[1]  # n_views
        contrast_feature = torch.cat(torch.unbind(features, dim=1), dim=0)

        if self.contrast_mode == 'one':
            anchor_feature = features[:, 0]
            anchor_count = 1
        elif self.contrast_mode == 'all':
            anchor_feature = contrast_feature
            anchor_count = contrast_count
        else:
            raise ValueError(f'Unknown mode: {self.contrast_mode}')

        # Compute logits
        anchor_dot_contrast = torch.div(
            torch.matmul(anchor_feature, contrast_feature.T),
            self.temperature)
        # For numerical stability
        logits_max, _ = torch.max(anchor_dot_contrast, dim=1, keepdim=True)
        logits = anchor_dot_contrast - logits_max.detach()

        # Tile mask
        mask = mask.repeat(anchor_count, contrast_count)
        # Mask-out self-contrast cases
        logits_mask = torch.scatter(
            torch.ones_like(mask),
            1,
            torch.arange(batch_size * anchor_count).view(-1, 1).to(device),
            0
        )
        mask = mask * logits_mask

        # Compute log_prob
        exp_logits = torch.exp(logits) * logits_mask
        log_prob = logits - torch.log(exp_logits.sum(1, keepdim=True) + 1e-9) # Add epsilon for stability

        # Compute mean of log-likelihood over positive pairs
        mask_pos_pairs_sum = mask.sum(1)
        mean_log_prob_pos = (mask * log_prob).sum(1) / (mask_pos_pairs_sum + 1e-9) # Add epsilon for stability

        # Loss
        loss = - (self.temperature / self.base_temperature) * mean_log_prob_pos
        loss = loss.view(anchor_count, batch_size).mean()

        return loss


def teacher_student_contrastive_kd_loss(student_features, teacher_features, temperature=0.1):
    """
    Contrastive Knowledge Distillation loss between a teacher and a student.
    Trains the student to have features similar to the teacher for the same sample,
    and dissimilar to the teacher's features for other samples.
    Assumes input features are of shape [batch_size, feature_dim].
    """
    device = student_features.device
    batch_size = student_features.shape[0]

    # L2 normalization (can be skipped if already done by the projection head)
    student_features_norm = F.normalize(student_features, p=2, dim=1)
    teacher_features_norm = F.normalize(teacher_features, p=2, dim=1)

    # Calculate similarity matrix: sim_matrix[i, j] is the similarity between student_i and teacher_j
    sim_matrix = torch.matmul(student_features_norm, teacher_features_norm.T) / temperature # Shape: [bsz, bsz]

    # The positive pairs are (student_i, teacher_i), so the diagonal elements are the correct "class"
    labels = torch.arange(batch_size).to(device)

    # Use cross-entropy loss to maximize similarity with the corresponding teacher feature
    loss = F.cross_entropy(sim_matrix, labels)
    return loss